import { BadRequestException, Injectable, NotFoundException } from '@nestjs/common';
import { CreateSubscriptionPlanDto } from './dto/create-subscription_plan.dto';
import { UpdateSubscriptionPlanDto } from './dto/update-subscription_plan.dto';
import { SubscriptionPlan } from './entities/subscription_plan.entity';
import { Repository } from 'typeorm';
import { InjectRepository } from '@nestjs/typeorm';
import { SubscriptionPlanUser } from './entities/subscription_plan_user.entity';
import { CreateSubscriptionPlanDtoUser } from './dto/create-subscription_plan_user.dto';
import { StatusPlan } from 'src/shared/enums/enum';
import { User } from 'src/shared/users/entities/user.entity';

@Injectable()
export class SubscriptionPlanService {
  constructor(
    @InjectRepository(SubscriptionPlan)
    private subscriptionPlanRepository: Repository<SubscriptionPlan>,
    @InjectRepository(SubscriptionPlanUser)
    private subscriptionPlanUserRepository: Repository<SubscriptionPlanUser>,
    @InjectRepository(User)
    private userRepository: Repository<User>,
  ) {}
  
  async create(createSubscriptionPlanDto: CreateSubscriptionPlanDto) {
    const existTypePlan = await this.subscriptionPlanRepository.findOne({
      where: {
        nome: createSubscriptionPlanDto.name_plan,
      },});
    if (existTypePlan){
      throw new BadRequestException('Este plano ja existe')
    }
    const newSubscriptionPlan = this.subscriptionPlanRepository.create({
      ...createSubscriptionPlanDto,
      nome: createSubscriptionPlanDto.name_plan // Mapeando 'name' para 'nome'
    });
    return this.subscriptionPlanRepository.save(newSubscriptionPlan);
  }



  async addUserToSubscriptionPlan(createSubscriptionPlanDtoUser: CreateSubscriptionPlanDtoUser) {
    const subscriptionPlan = await this.subscriptionPlanRepository.findOne({
      where: { id: createSubscriptionPlanDtoUser.subscriptionPlanId},
    });

    if (!subscriptionPlan) {
      throw new NotFoundException(`Plano de assinatura com ID ${createSubscriptionPlanDtoUser.subscriptionPlanId} não encontrado.`);
    }

    const { user_name, email, cnpj } = createSubscriptionPlanDtoUser;

    if (!user_name && !email && !cnpj) {
      throw new BadRequestException('É necessário fornecer nome de usuário, email ou CNPJ para encontrar o usuário.');
    }

    const userToFind = await this.userRepository.findOne({
      where: [
        { user_name: user_name },
        { email: email },
        { document: cnpj }
      ].filter(c => Object.values(c)[0]) // Filtra critérios vazios
    });

    if (!userToFind) {
      throw new NotFoundException('Usuário não encontrado com os dados fornecidos.');
    }
    
    const enddate = new Date();
    enddate.setDate(enddate.getDate() + subscriptionPlan.duration);

    const newUserPlan = this.subscriptionPlanUserRepository.create({
      subscriptionPlan,
      user: userToFind,
      start_date: new Date(),
      end_date: enddate,
      status: StatusPlan.ACTIVE,
    });

    this.subscriptionPlanUserRepository.save(newUserPlan)

    return 'Plano de assinatura adicionado com sucesso!' ;


  }


  async findAllUsersWhithPlan() {
    const userswhithplan = await this.subscriptionPlanUserRepository.createQueryBuilder('subscriptionPlanUser')
    .leftJoin('subscriptionPlanUser.subscriptionPlan', 'subscriptionPlan')
    .leftJoin('subscriptionPlanUser.user', 'user')
    .select('subscriptionPlan', 'subscriptionPlan')
    .addSelect('user.user_name', 'name_user')
    .addSelect('user.email', 'email_user')
    .addSelect('user.id', 'id_user')
    .addSelect('subscriptionPlanUser.start_date', 'start_date')
    .addSelect('subscriptionPlanUser.end_date', 'end_date')
    .addSelect('subscriptionPlanUser.status', 'status')
    .addSelect('subscriptionPlanUser.id', 'id_plan_user')
    .addSelect('subscriptionPlan.price', 'price')
    .addSelect('subscriptionPlan.duration', 'duration')
    .addSelect('subscriptionPlan.nome', 'name_plan')

    .getRawMany();

    const totalRevenue = userswhithplan.reduce((sum, item) => {
      return sum + (Number(item.price) || 0);
    }, 0);
    const totalActive = userswhithplan.filter(item => item.status === StatusPlan.ACTIVE).length;

    const newUserThisMonth = userswhithplan.filter(item => item.start_date.getMonth() === new Date().getMonth()).length;



    return { userswhithplan, othersInfo:{ totalActive, totalRevenue, newUserThisMonth} };
  }

  async findOneUserWhithPlan(id: number) {
    const existUserWhithPlan = await this.subscriptionPlanUserRepository.createQueryBuilder('subscriptionPlanUser')
    .where('subscriptionPlanUser.id = :id', { id })
    .leftJoin('subscriptionPlanUser.subscriptionPlan', 'subscriptionPlan')
    .leftJoin('subscriptionPlanUser.user', 'user')
    .select('subscriptionPlan', 'subscriptionPlan')
    .addSelect('user.user_name', 'name_user')
    .addSelect('user.email', 'email_user')
    .addSelect('user.id', 'id_user')
    .addSelect('subscriptionPlanUser.start_date', 'start_date')
    .addSelect('subscriptionPlanUser.end_date', 'end_date')
    .addSelect('subscriptionPlanUser.status', 'status')
    .addSelect('subscriptionPlanUser.id', 'id_plan_user')
    .getRawOne();



    if(!existUserWhithPlan){
      throw new BadRequestException('Ops, Usuario sem plano')
    }

    return existUserWhithPlan;

  }

  async findAll() {
    return this.subscriptionPlanRepository.find();
  }



  update(id: number, updateSubscriptionPlanDto: UpdateSubscriptionPlanDto) {
    return `This action updates a #${id} subscriptionPlan`;
  }

  remove(id: number) {
    return `This action removes a #${id} subscriptionPlan`;
  }
}
