import {
  ConflictException,
  Inject,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { CreateFilesDto } from './dto/create-files.dto';
import { UpdateFilesDto } from './dto/update-files.dto';
import { CreateFolderDto } from './dto/create-path.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Files } from './entities/files.entity';
import { Repository, DeepPartial, IsNull, Like } from 'typeorm';
import { unlink } from 'fs/promises';
import { join } from 'path';
import { Folder } from './entities/path.entity';
import { UpdatePathDto } from './dto/update-path.dto';

@Injectable()
export class FilesService {
  constructor(
    @InjectRepository(Files)
    private filesRepository: Repository<Files>,

    @InjectRepository(Folder)
    private folderRepository: Repository<Folder>,
  ) {}

  async create(file: Express.Multer.File, createFilesDto: CreateFilesDto) {
    // 1. Buscamos a pasta (folder) pelo ID que veio no DTO
    // (Assumindo que você ajustou o JSON para enviar 'folder': 35)
    const folderEntity = await this.folderRepository.findOne({
      where: { id: Number(createFilesDto.folder) },
    });

    if (!folderEntity) {
      throw new NotFoundException('A pasta informada não existe.');
    }

    // 2. Criamos o arquivo associando a entidade da pasta encontrada
    const newFile = this.filesRepository.create({
      file_name: file.originalname,
      name: createFilesDto.file_name,
      version: createFilesDto.version,
      description: createFilesDto.description,
      path: file.path,
      type: file.mimetype,
      size: file.size,
      folder: folderEntity,
    } as DeepPartial<Files>);

    await this.filesRepository.save(newFile);

    return 'Arquivo salvo com sucesso!';
  }

  async createFolder(createFolderDto: CreateFolderDto): Promise<Folder> {
    console.log('SERVICE CHAMADA!', createFolderDto);

    const { name, path } = createFolderDto;
    // Garante que 'Admin' sempre esteja presente e remove duplicatas.
    const allowedRoles = [
      ...new Set([
        'admin',
        'admin-ecommerce',
        'admin-cursos',
        'admin-monitoramento',
        ...(createFolderDto.allowedRoles || []),
      ]),
    ];

    // 🛠️ Função auxiliar para limpar barras duplas e garantir barra inicial
    const sanitizePath = (p: string) => {
      if (!p) return '/';
      // Substitui múltiplas barras por uma só e garante que começa com /
      return `/${p
        .split('/')
        .filter((x) => x)
        .join('/')}`;
    };

    // 👉 1. Lógica para criar na raiz (se path for vazio ou apenas "/")
    if (!path || path.trim() === '' || path === '/') {
      const exists = await this.folderRepository.findOne({
        where: { name, parent: IsNull() },
      });

      if (exists) return exists;

      const newFolder = this.folderRepository.create({
        name,
        path: `/`, // 💡 CORREÇÃO: O path da raiz deve ser /nome
        allowedRoles,
        parent: undefined,
      });

      return await this.folderRepository.save(newFolder);
    }

    // 👉 2. Criar estrutura de pastas baseada no path
    const parts = path.split('/').filter((p) => p.trim() !== '');
    let parent: Folder | null = null;
    let currentPath = '';

    for (const part of parts) {
      // 💡 CORREÇÃO: Constrói o path sempre limpo: /pasta1/pasta2
      currentPath = `${currentPath}/${part}`;

      let folder = await this.folderRepository.findOne({
        where: {
          name: part,
          parent: parent ? { id: parent.id } : IsNull(),
        },
      });

      if (!folder) {
        folder = this.folderRepository.create({
          name: part,
          path: currentPath, // Salva como /pasta8
          allowedRoles,
          parent: parent || undefined,
        });

        folder = await this.folderRepository.save(folder);
      }
      parent = folder;
    }

    // 👉 3. Cria a pasta final solicitada
    // 💡 CORREÇÃO: Garante que não duplica a barra se o pai for raiz
    const parentPathStr = parent?.path === '/' ? '' : parent?.path || '';
    const finalPath = `${parentPathStr}/${name}`;

    const existsFinal = await this.folderRepository.findOne({
      where: {
        name,
        parent: parent ? { id: parent.id } : IsNull(),
      },
    });

    if (existsFinal) return existsFinal;

    const finalFolder = this.folderRepository.create({
      name,
      path: finalPath, // Agora será algo como /pasta8/pasta9
      allowedRoles,
      parent: parent || undefined,
    });

    return await this.folderRepository.save(finalFolder);
  }

  async getRaiz(currentUser: any): Promise<Folder[]> {
    // Para 'simple-array', usamos 'Like' para verificar se a role do usuário está contida na lista de roles permitidas.
    return this.folderRepository.find({
      where: {
        parent: IsNull(),
        allowedRoles: Like(`%${currentUser.roles}%`),
      },
    });
  }

  async getRelation(pastaRaizId: number, currentUser: any) {
    const parentFolder = await this.folderRepository.findOne({
      where: { id: pastaRaizId, allowedRoles: Like(`%${currentUser.roles}%`) },
      relations: ['children', 'files'],
    });

    // Se a pasta pai não for encontrada, lança um erro 404.
    if (!parentFolder) {
      throw new NotFoundException(
        `Pasta com ID ${pastaRaizId} não encontrada.`,
      );
    }

    // 2. Retorna um objeto contendo apenas as subpastas e os arquivos.
    // O frontend terá acesso a `children` e `files`.
    const { children, files: originalFiles } = parentFolder;

    // Mapeia os arquivos para remover a propriedade 'path' de cada um.
    const files = originalFiles.map((file) => {
      const { path, ...fileData } = file; // 'path' é extraído e o resto ('fileData') é mantido.
      return { fileData };
    });

    return { children, files };
  }

  async deleteFolder(id: number): Promise<void> {
    // Carrega a pasta com os filhos
    const folder = await this.folderRepository.findOne({
      where: { id },
      relations: ['children'],
    });

    if (!folder) {
      throw new NotFoundException('Pasta não encontrada.');
    }

    await this.folderRepository.remove(folder);
  }

  async editPath(id: number, updateFolderDto: UpdatePathDto) {
    const folder = await this.folderRepository.findOne({ where: { id } });

    if (!folder) {
      throw new NotFoundException(
        `Não foi possível editar a pasta com ID ${id}`,
      );
    }

    const oldPath = folder.path;

    //
    // 👉 Recalcular novo path apenas se nome mudar
    //
    let newPath = oldPath;

    if (updateFolderDto.name) {
      const parts = oldPath.split('/').filter(Boolean); // remove strings vazias
      parts[parts.length - 1] = updateFolderDto.name;
      newPath = '/' + parts.join('/');
    }

    //
    // 👉 Atualizar a pasta principal
    //
    await this.folderRepository.update(id, {
      name: updateFolderDto.name ?? folder.name,
      path: newPath,
      // Garante que 'Admin' sempre esteja presente e remove duplicatas.
      allowedRoles: updateFolderDto.allowedRoles
        ? [
            ...new Set([
              'admin',
              'admin-ecommerce',
              'admin-cursos',
              'admin-monitoramento',
              ...updateFolderDto.allowedRoles,
            ]),
          ]
        : folder.allowedRoles,
    });

    //
    // 👉 Atualizar paths de todas as pastas filhas
    //
    const children = await this.folderRepository.query(
      `SELECT * FROM folders WHERE path LIKE ? AND id != ?`,
      [`${oldPath}%`, id],
    );

    for (const child of children) {
      const updatedPath = child.path.replace(oldPath, newPath);
      await this.folderRepository.update(child.id, { path: updatedPath });
    }

    return { oldPath, newPath };
  }

  async findAll(): Promise<Files[]> {
    return this.filesRepository.find({
      select: {
        id: true,
        file_name: true,
        name: true,
        version: true,
        description: true,
        type: true,
        size: true,
        created_on: true,
      },
      order: { created_on: 'DESC' },
    });
  }

  async getFilePath(id: number) {
    const file = await this.filesRepository.findOne({
      where: { id },
    });

    if (!file) {
      throw new NotFoundException('Arquivo não encontrado');
    }

    return file.path;
  }

  async remove(id: number): Promise<boolean> {
    const files = await this.filesRepository.findOneBy({ id });

    if (!files) {
      throw new NotFoundException('Arquivo não encontrado');
    }

    const path = join(files.path);

    try {
      await unlink(path);
    } catch (error) {
      throw new NotFoundException(
        { message: 'Arquivo não encontrado no disco...' },
        error.message,
      );
    }

    await this.filesRepository.delete(id);

    return true;
  }
}
