import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards, Dependencies, Put } from '@nestjs/common';
import { EquipamentsService } from './equipaments.service';
import { UpdateEquipamentDto, UpdateSchedulesByGroupDto, UpdateSchedulesDto, UpdateStatusDto } from './dto/update-equipament.dto';
import { AuthGuard } from 'src/shared/auth/auth.guard';
import { PermissionsGuard } from 'src/shared/auth/permissions.guards';
import { RequiredRoles } from 'src/shared/auth/required-roles.decorator';
import { Roles } from 'src/shared/enums/enum';
import { RoleGuard } from 'src/shared/auth/role/role.guard';
import { ApiBearerAuth, ApiOperation, ApiResponse, ApiTags } from '@nestjs/swagger';
import { CreateEquipamentFirstConnectionDto } from './dto/create-equipament.dto';
import { CurrentUser } from 'src/shared/users/current.user.decorator';
import { AddEquipamentToGroupDto } from './dto/add-equipament-to-group.dto';

@ApiTags('Equipamentos')
@Controller('equipamentos')
export class EquipamentsController {
  constructor(private readonly equipamentsService: EquipamentsService) {}

  @Post('/primeiro-registro/')
  async createFirstConnection(@Body() dto: CreateEquipamentFirstConnectionDto) {
    return this.equipamentsService.createFirstConnection(dto);
  }

  @Post('/grupo')
  @UseGuards(AuthGuard)
  async addEquipamentToGroup(@CurrentUser() currentUser: any, @Body() addEquipamentToGroupDto: AddEquipamentToGroupDto) {
    return this.equipamentsService.addEquipamentToGroup(currentUser.id, addEquipamentToGroupDto);
  }

  @Get('/')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMVISO360, Roles.ADMIN)
  countAll() {
    return this.equipamentsService.countAll();
  }

  @Get('/lista-todos')
  findAll() {
    return this.equipamentsService.findAll();
  }

  @Get('/ping/:mac')
  ping(@Param('mac') mac: string) {
    return this.equipamentsService.ping(mac);
  }

  @Get('/programar-individual/:equipId/usuario/:id')
  findSchedulesByEquipament(@Param('equipId') equipId: string, @Param('id') id: string) {
    return this.equipamentsService.findSchedulesByEquipament(+equipId, +id);
  }

  @UseGuards(AuthGuard, PermissionsGuard)
  @Get('/observados/:id')
  findMonitoredEquipaments(@Param('id') id: string) { 
    return this.equipamentsService.findMonitoredEquipaments(+id);
  }

  @Get('/cards-admin')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMVISO360, Roles.ADMIN)
  countForCards() {
    return this.equipamentsService.countForCards();
  }

  @Get('/usuario/:id')
  @UseGuards(AuthGuard, PermissionsGuard)
  findAllByUser(@Param('id') id: string) {
    return this.equipamentsService.findAllByUser(+id);
  }

  @Get(':id/:type')
  @UseGuards(AuthGuard)
  @RequiredRoles(Roles.ADMVISO360, Roles.ADMIN)
  findOne(@Param('id') id: number, @Param('type') type: string ) {
    return this.equipamentsService.findOne(+id, type?.toLowerCase() ?? '' );
  }

  @Get(':type')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMVISO360, Roles.ADMIN)
  countByType(@Param('type') type: string) {
    return this.equipamentsService.countByType(type?.toLowerCase() ?? '');
  }

  @Patch('/status')
  updateStatus(@Body() dto: UpdateStatusDto) {
    return this.equipamentsService.updateStatus(dto);
  }

  @Patch(':equipId/usuario/:id')
  @UseGuards(AuthGuard, PermissionsGuard)
  update(@Param('equipId') equipId: string, @Param('id') id: string, @Body() updateEquipamentDto: UpdateEquipamentDto) {
    return this.equipamentsService.update(+equipId, +id, updateEquipamentDto);
  }

  @Patch('desvincular/:equipId/usuario/:id')
  @UseGuards(AuthGuard, PermissionsGuard)
  unlinkDeviceByUser(@Param('equipId') equipId: string, @Param('id') id: string) {
    return this.equipamentsService.unlinkDeviceByUser(+equipId, +id);
  }

  @UseGuards(AuthGuard, PermissionsGuard)
  @Patch('programar-individual/:equipId/usuario/:id')
  updateScheduleByEquipament(@Param('equipId') equipId: string, @Param('id') id: string,@Body() updateSchedulesDto: UpdateSchedulesDto) {
    return this.equipamentsService.updateScheduleByEquipament(+equipId, +id, updateSchedulesDto);
  }

  @UseGuards(AuthGuard)
  @Patch('programar-em-grupo/')
  updateScheduleByGroup(@CurrentUser() currentUser: any, @Body() updateScheduleByGroupDto: UpdateSchedulesByGroupDto) {
    return this.equipamentsService.updateScheduleByGroup(currentUser.id, updateScheduleByGroupDto);
  }

  @UseGuards(AuthGuard)
  @Put('/grupo')
  removeEquipamentFromGroup(@CurrentUser() currentUser: any, @Body() removeEquipamentFromGroupDto: AddEquipamentToGroupDto) {
    return this.equipamentsService.removeEquipamentFromGroup(currentUser.id, removeEquipamentFromGroupDto);
  }

  @UseGuards(AuthGuard, PermissionsGuard)
  @Delete(':equipId/usuario/:id/')
  remove(@Param('id') id: string, @Param('equipId') equipId: string) {
    return this.equipamentsService.remove(+equipId, +id);
  }
}
