import {
  BadRequestException,
  ConflictException,
  forwardRef,
  Inject,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { MoreThan, Repository } from 'typeorm';
import * as bcrypt from 'bcrypt';
import { CreateUserDto } from './dto/create-user.dto';
import { UpdateUserDto, UpdateUserPasswordDto } from './dto/update-user.dto';
import { User } from './entities/user.entity';
import { Equipament } from 'src/visoflex360/equipaments/entities/equipament.entity';
import { Door } from 'src/visoflex360/doors/entities/door.entity';
import { AuthService } from 'src/shared/auth/auth.service';
import { PasswordResetToken } from './entities/password-reset-token.entity';
import { EventEmitter2 } from '@nestjs/event-emitter';
import { ResetPasswordDto } from './dto/recover-password.dto';
import { CryptoUtil } from 'src/utils/encrypt';

@Injectable()
export class UsersService {
  constructor(
    @InjectRepository(User)
    private usersRepository: Repository<User>,

    @InjectRepository(PasswordResetToken)
    private resetTokenRepository: Repository<PasswordResetToken>,

    @InjectRepository(Equipament)
    private equipamentsRepository: Repository<Equipament>,

    @InjectRepository(Door)
    private doorsRepository: Repository<Door>,

    @Inject(forwardRef(() => AuthService))
    private readonly authService: AuthService,

    private readonly eventEmitter: EventEmitter2,
  ) {}

  async create(createUserDto: CreateUserDto) {
    const userExist = await this.findOneByEmail(createUserDto.email);
    if (userExist) {
      return { message: ['E-mail já cadastrado'] };
    }

    let sincCode = '#' + Math.floor(10000 + Math.random() * 90000);

    while (
      (await this.usersRepository.findOneBy({ sinc_code: sincCode })) !== null
    ) {
      sincCode = '#' + Math.floor(10000 + Math.random() * 90000);
    }

    const password = await bcrypt.hash(createUserDto.password, 10);
    const user = await this.usersRepository.create({
      ...createUserDto,
      password,
      sinc_code: sincCode,
    });

    const userCreated = await this.usersRepository.save(user);

    if (!userCreated) {
      throw new BadRequestException('Erro durante a criação da conta');
    }

    return this.authService.login(userCreated);
  }

  async forgotPassword(email: string) {
    const user = await this.usersRepository.findOneBy({ email });

    if (!user) {
      return {
        message:
          'Se o e-mail estiver cadastrado, um código de recuperação será enviado.',
      };
    }

    await this.resetTokenRepository.update(
      { user: { id: user.id } },
      { used: true },
    );

    const code = Math.floor(100000 + Math.random() * 900000).toString();
    const expires_at = new Date();
    expires_at.setMinutes(expires_at.getMinutes() + 15);

    const resetToken = this.resetTokenRepository.create({
      user: user,
      code,
      expires_at,
    });

    await this.resetTokenRepository.save(resetToken);

    this.eventEmitter.emit('forgot.password', {
      cellphone: user.cellphone,
      code,
    });

    return {
      message:
        'Se o e-mail estiver cadastrado, um código de recuperação será enviado.',
    };
  }

  async findMyProfile(id: number) {
    const user = await this.findOne(id);

    if (!user) {
      throw new NotFoundException('Usuário não encontrado');
    }

    return {
      id: user.id,
      document: user.document,
      company_name: user.company_name,
      user_name: user.user_name,
      email: user.email,
      roles: user.roles,
      cellphone: user.cellphone,
      maintanance_notf: user.maintanance_notf,
      alert_notf: user.alert_notf,
      sinc_code: user.sinc_code,
    };
  }

  findOne(id: number) {
    return this.usersRepository.findOneBy({ id });
  }

  findOneByEmail(email: string) {
    return this.usersRepository.findOneBy({ email });
  }

  async findResume() {
    const users = await this.usersRepository.find();

    const hasEquipament = await Promise.all(
      users.map(async (user) => {
        const equipaments = await this.equipamentsRepository.find({
          where: { user: { id: user.id } },
        });

        const hasAlert = await this.doorsRepository.count({
          where: {
            has_alert: true,
            equipament: {
              user: {
                id: user.id,
              },
            },
          },
          relations: {
            equipament: {
              user: true,
            },
          },
        });

        return {
          id: user.id,
          nome_empresa: user.company_name,
          documento: user.document,
          nome_cliente: user.user_name,
          sinc_code: user.sinc_code,
          email: user.email,
          celular: user.cellphone,
          total_equipamentos: equipaments.length,
          alertas_portas: hasAlert,
          status: user.is_aproved,
          role: user.roles,
        };
      }),
    );
    return hasEquipament;
  }

  async countAll() {
    const users = await this.usersRepository.count();
    return users;
  }

  async resetPassword(resetDto: ResetPasswordDto) {
    const { email, code, newPassword } = resetDto;

    const resetToken = await this.resetTokenRepository.findOne({
      where: {
        code,
        used: false,
        expires_at: MoreThan(new Date()),
        user: { email },
      },
      relations: ['user'],
    });

    if (!resetToken) {
      throw new BadRequestException(
        'Código de recuperação inválido ou expirado.',
      );
    }

    const user = resetToken.user;
    user.password = await bcrypt.hash(newPassword, 10);
    await this.usersRepository.save(user);

    resetToken.used = true;
    await this.resetTokenRepository.save(resetToken);

    return { message: 'Senha redefinida com sucesso.' };
  }

  async changeUserPassword(id, updateUserPasswordDto: UpdateUserPasswordDto) {
    const user = await this.usersRepository.findOneBy({ id });

    if (
      user &&
      bcrypt.compareSync(updateUserPasswordDto.current_password, user.password)
    ) {
      try {
        user.password = await bcrypt.hash(
          updateUserPasswordDto.new_password,
          10,
        );
        await this.usersRepository.update(user.id, user);
        return 'Senha alterada com sucesso';
      } catch (error) {
        throw new BadRequestException({
          message: 'Erro ao tentar trocar a senha do usuário.' + error,
        });
      }
    }
    throw new BadRequestException('Erro ao tentar alterar a senha');
  }

  async updateUserStatus(id: number, aproved: boolean) {
    const user = await this.usersRepository.findOneBy({ id });

    if (!user) {
      throw new NotFoundException('Usuário não encontrado');
    }

    let message: string;
    aproved
      ? ((user.is_aproved = true),
        (message =
          'Status do código de sincronização do usuário alterado para aprovado'))
      : ((user.is_aproved = false),
        (message =
          'Status do código de sincronização do usário alterado para reprovado'));

    try {
      user.is_aproved = user.is_aproved != aproved ? aproved : user.is_aproved;
      await this.usersRepository.save(user);
    } catch (error) {
      throw new BadRequestException(
        'Erro ao tentar mudar o status do usuário. \n Erro: ' + error,
      );
    }

    return message;
  }

  async update(id: number, updateUserDto: UpdateUserDto) {
    const user = await this.usersRepository.findOneBy({ id });

    if (!user) {
      throw new NotFoundException('Usuário não encontrado');
    }

    if (updateUserDto.password) {
      try {
        const hashedPassword = await bcrypt.hash(updateUserDto.password, 10);
        updateUserDto.password = hashedPassword;
      } catch (error) {
        throw new BadRequestException('Erro ao atualizar senha');
      }
    }

    try {
      this.usersRepository.merge(user, updateUserDto);
      await this.usersRepository.save(user);
    } catch (error) {
      throw new BadRequestException(
        'Erro durante a atualização dos dados do usuário. ' + error,
      );
    }

    return 'Usuário atualizado com sucesso';
  }

  async remove(id: number) {
    const user = await this.usersRepository.findOneBy({ id });
    if (!user) {
      throw new NotFoundException({ message: 'Usuário não encontrado' });
    }

    try {
      await this.usersRepository.remove(user);
    } catch (error) {
      console.log(error);
      throw new BadRequestException(
        'Erro ao tentar deletar o usuário. ' + error,
      );
    }
    return { message: 'Usuário deletado com sucesso' };
  }

  /* Funções Auxiliares */

  async findUserById(userId: number) {
    const user = await this.usersRepository.findOneBy({ id: userId });

    if (!user) {
      throw new NotFoundException('Usuário não encontrado');
    }

    return user;
  }

  async findUserWithoutRestricData(id: any) {
    const user = await this.usersRepository.findOne({
      where: { ...id },
      select: [
        'id',
        'document',
        'company_name',
        'user_name',
        'email',
        'roles',
        'cellphone',
        'maintanance_notf',
        'alert_notf',
        'sinc_code',
        'is_aproved',
      ],
    });

    if (!user) {
      throw new NotFoundException('Usuário não encontrado');
    }

    return user;
  }
}
