import { AuthGuard } from 'src/shared/auth/auth.guard';
import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards, ValidationPipe, Put, Query, ParseBoolPipe } from '@nestjs/common';
import { UsersService } from './users.service';
import { CreateUserDto, CreateUserResponseDto } from './dto/create-user.dto';
import { UpdateUserDto, UpdateUserPasswordDto } from './dto/update-user.dto';
import { RequiredRoles } from 'src/shared/auth/required-roles.decorator';
import { Roles } from '../enums/enum';
import { RoleGuard } from 'src/shared/auth/role/role.guard';
import { PermissionsGuard } from 'src/shared/auth/permissions.guards';
import { ApiBearerAuth, ApiTags } from '@nestjs/swagger';
import { forgotPasswordDto, ResetPasswordDto } from './dto/recover-password.dto';
import { CurrentUser } from './current.user.decorator';

@Controller('usuarios')
export class UsersController {
  constructor(private readonly usersService: UsersService) {}

  @Post('/registro')
  create(@Body(new ValidationPipe({whitelist: true})) createUserDto: CreateUserDto) {
    return this.usersService.create(createUserDto);
  }

  @Post('esqueceu-senha')
    forgotPassword(@Body() forgotPasswordDto: forgotPasswordDto) {
      return this.usersService.forgotPassword(forgotPasswordDto.email);
  }

  @Get('/me')
  @UseGuards(AuthGuard)
  findUserWithoutRestricData(@CurrentUser() currentUser: any) {
    return this.usersService.findUserWithoutRestricData(currentUser.id);
  }

  @Get('/')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN)
  findResume(){
    return this.usersService.findResume();
  }

  @Get('/contagem-total')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMVISO360, Roles.ADMIN)
  countAll(){
    return this.usersService.countAll();
  }
  
  @Get('my-profile/:id')
  @UseGuards(AuthGuard, PermissionsGuard)
  findMyProfile(@Param('id') id: string){
    return this.usersService.findMyProfile(+id);
  }

  @Get(':id')
  @UseGuards(AuthGuard, PermissionsGuard)
  findOne(@Param('id') id: string) {
    return this.usersService.findOne(+id);
  }



  @Put(':id/')
  @UseGuards(AuthGuard)
  @RequiredRoles(Roles.ADMVISO360, Roles.ADMIN)
  updateUserStatus(@Param('id') id: string, @Query('aproved', new ParseBoolPipe({ errorHttpStatusCode: 400 })) aproved: boolean) {
    return this.usersService.updateUserStatus(+id, aproved);
  }

  @Patch('alterar-senha')
    resetPassword(@Body() resetPasswordDto: ResetPasswordDto) {
      return this.usersService.resetPassword(resetPasswordDto);
  }

  @Patch('change-password/:id')
  @UseGuards(AuthGuard, PermissionsGuard)
  changeUserPassword(@Param('id') id : string, @Body() updateUserPasswordDto : UpdateUserPasswordDto) {
  return this.usersService.changeUserPassword(+id, updateUserPasswordDto);
  }
  
  @Patch(':id')
  @UseGuards(AuthGuard, PermissionsGuard)
  update(@Param('id') id: string, @Body() updateUserDto: UpdateUserDto) {
    return this.usersService.update(+id, updateUserDto);
  }

  @Delete(':id')
  @UseGuards(AuthGuard, PermissionsGuard)
  @RequiredRoles(Roles.ADMVISO360, Roles.ADMIN)
  remove(@Param('id') id: string) {
    return this.usersService.remove(+id);
  }
}
