import { PartialType } from '@nestjs/mapped-types';
import { CreateUserDto } from './create-user.dto';
import {
  IsBoolean,
  IsBooleanString,
  IsEmail,
  IsEnum,
  IsNotEmpty,
  IsObject,
  IsOptional,
  IsString,
  IsStrongPassword,
  MinLength,
} from 'class-validator';
import { ApiProperty } from '@nestjs/swagger';
import { Transform } from 'class-transformer';
import { Roles } from 'src/shared/enums/enum';

export class UpdateUserDto {
  @Transform(({ value }) => value.toLowerCase())
  @IsString({ message: 'O nome da empresa deve ser uma string.' })
  @IsNotEmpty({ message: 'O nome da empresa não pode estar vazio.' })
  @MinLength(3, {
    message: 'O nome da empresa deve ter no mínimo 3 caracteres.',
  })
  company_name: string;

  @Transform(({ value }) => value.toLowerCase())
  @IsString({ message: 'O nome deve ser uma string.' })
  @IsNotEmpty({ message: 'O nome não pode estar vazio.' })
  @MinLength(3, { message: 'O nome deve ter no mínimo 3 caracteres.' })
  user_name: string;

  @IsOptional()
  @IsString({ message: 'O documento deve ser uma string.' })
  document: string;

  @Transform(({ value }) => value.toLowerCase())
  @IsEmail({}, { message: 'Por favor, forneça um endereço de e-mail válido.' })
  @IsNotEmpty({ message: 'O e-mail não pode estar vazio.' })
  email: string;

  @IsOptional()
  @IsString({ message: 'O celular deve ser uma string.' })
  cellphone: string;

  @IsOptional()
  maintanance_notf: boolean;

  @IsOptional()
  alert_notf: boolean;

  @IsOptional()
  is_aproved: boolean;

  @IsOptional()
  @IsEnum(Roles)
  roles: Roles;

  @IsOptional()
  @IsString({ message: 'O número do cartão de crédito deve ser uma string.' })
  credit_card_number: string;

  @IsOptional()
  @IsString({
    message: 'O mês de expiração do cartão de crédito deve ser uma string.',
  })
  credit_card_expiry_month: string;

  @IsOptional()
  @IsString({
    message: 'O ano de expiração do cartão de crédito deve ser uma string.',
  })
  credit_card_expiry_year: string;

  @IsOptional()
  @IsString({
    message: 'O nome do titular do cartão de crédito deve ser uma string.',
  })
  credit_card_holder_name: string;

  @IsOptional()
  @IsString({
    message:
      'O código de segurança (CCV) do cartão de crédito deve ser uma string.',
  })
  ccv: string;

  @IsOptional()
  @IsString()
  password: string;
}

export class UpdateUserPasswordDto {
  @IsString()
  current_password: string;

  @ApiProperty({
    description: 'O campo de senha do usuário',
    example: '123456',
  })
  @IsNotEmpty({ message: 'O campo senha é obrigatório' })
  @IsStrongPassword(
    { minLength: 6, minLowercase: 0, minUppercase: 0, minSymbols: 0 },
    { message: 'A senha deve conter pelo menos 6 caracteres' },
  )
  new_password: string;
}
