import { HttpService } from '@nestjs/axios';
import { Injectable, Logger, PayloadTooLargeException } from '@nestjs/common';
import { SendMessageDto } from '../presentation/dto/send-message.dto';
import { firstValueFrom } from 'rxjs';
import { OnEvent } from '@nestjs/event-emitter';

@Injectable()
export class MessagingService {
    private readonly urlApiGzappy = process.env.GZAPPY_API_URL;
    private readonly bearerToken = process.env.GZAPPY_BEARER_TOKEN;
    private readonly publicId = process.env.GZAPPY_PUBLIC_ID;
    private readonly adminCellphone = process.env.ADMIN_CELLPHONE; 
    private readonly logger = new Logger(MessagingService.name);
     
    constructor(
        private readonly httpService: HttpService
    ) {}
    async sendWhatsappText(sendMessageDto: SendMessageDto): Promise<void> {

        if (!this.urlApiGzappy || !this.bearerToken || !this.publicId) {
            this.logger.error('Credenciais da Gzappy (URL, Bearer Token ou Public ID ) não estão configuradas.');
            return;
        }

        const payload= {
            phone: sendMessageDto.cellphone,
            message: sendMessageDto.message
        }

        const headers = {
        Authorization: `Bearer ${this.bearerToken}`,
        'Content-Type': 'application/json',
        };

        try {
            await firstValueFrom(
                this.httpService.post(this.urlApiGzappy+'/message/send-text', payload, { headers }),
            );
            this.logger.log(`Mensagem de WhatsApp enviada com sucesso para ${sendMessageDto.cellphone}`);
        } catch (error) {
            this.logger.error(
                `Falha ao enviar mensagem de WhatsApp para ${sendMessageDto.message}`,
                error.response?.data || error.message,
            );
            throw new Error
        }
    }

    // ### Funções auxiliares e ouvintes de eventos ###

    @OnEvent('user.created')
    async handleUserCreated({ userName, cellphone }) {
        if (!userName || !cellphone) return;

        const message = `Olá, ${userName}! Boas notícias!\n\n` +
                        `🎉 Sua conta na plataforma de pedidos das lojas Croissant & CIA acaba de ser criada com sucesso!\n\n` +
                        `Continue conosco em https://www.pedidoscroissant.com.br e conheça nossos salgados geniais. `;

        await this.sendWhatsappText({
            cellphone: cellphone,
            message,
        });
    }

    @OnEvent('order.accepted_awaiting_payment')
    async handleOrderAccepted({ cellphone }) {
        if (!cellphone) return;

        const message = `A loja já aprovou o seu pedido na plataforma Pedidos Croissant\n\n` +
                        `Finalize o pagamento acessando sua conta em https://pedidoscroissant.com.br e em breve poderá deliciar-se com nossos maravilhosos salgados geniais`;

        
        await this.sendWhatsappText({
            cellphone: cellphone,
            message
        });
    }

    @OnEvent('quotation.accepted')
    async handleQuotationAccepted({ cellphone }) {
        if (!cellphone) return;

        const message = `Seu pedido foi aceito e já chamamos um entregador para leva-lo até você o mais rápido o possível\n\n` +
                        `Assim que seu pedido sair para entrega iremos avisa-lo.`;
    
        await this.sendWhatsappText({
            cellphone,
            message
        });
    }

    @OnEvent('order.reject')
    async handleOrderRejected({ cellphone, message}) {
        if (!cellphone) return

        await this.sendWhatsappText({cellphone, message})
    }

    @OnEvent('notify.store')
    async handleNotifyStore({ cellphone, message}) {
        if (!cellphone || !message) return;

        await this.sendWhatsappText({
            cellphone,
            message
        })
    }
    
    @OnEvent('forgot.password')
    async handleForgorPassword({ cellphone, code}) {
        if (!cellphone || !code) return

        const message = `Recebemos um pedido de troca senha para o seu e-mail.\n\n`+
                        `Na página de recuperação de senha digite esse código *${code}*\n\n`+
                        `IMPORTANTE: Se você não reconhece essa solicitação considere acessar sua conta e modificar a senha, além de entrar em contato conosco.`

        await this.sendWhatsappText({
            cellphone,
            message
        })
    }

    // @OnEvent('lower.stock')
    // async handleLowerStock({productsName}) {
    //     if (!this.adminCellphone) {
    //         this.logger.error('ADMIN_CELLPHONE não está configurado.');
    //         return;
    //     }
    
    //     const message = `Estes produtos estão com o estoque baixo: \n - ${productsName.join(' \n - ')}`

    //     await this.sendWhatsappText({
    //         cellphone: this.adminCellphone,
    //         message
    //     })
    // }

    @OnEvent('need.maintanance')
    async handleNeedMaintanance({cellphone, message}) {
        if (!cellphone || !message) return;

        await this.sendWhatsappText({
            cellphone,
            message
        });
    }
}