import { BadRequestException, ConflictException, Injectable, NotFoundException } from '@nestjs/common';
// import { CreateSignatureDto } from './dto/create-signature.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Class } from './entities/class.entity';
import { In, Repository } from 'typeorm';
// import { UpdateSignatureDto } from './dto/update-signature.dto';
import { diskStorage } from 'multer';
import * as multerS3 from 'multer-s3';
import { S3Client, PutObjectCommand } from '@aws-sdk/client-s3';
import * as sharp from 'sharp';
import { CreateClassDto } from './dto/create-video.dto';
import { UpdateClassDto } from './dto/upload-video.dto';


@Injectable()
export class ClassesService {
  constructor(
    @InjectRepository(Class)
    private classesRepository: Repository<Class>

    
  ) {}

   private s3 = new S3Client({
    region: process.env.AWS_S3_REGION!,
    credentials: {
      accessKeyId: process.env.AWS_ACCESS_KEY_ID!,
      secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY!,
    },
  });



 async create(createClassDto: CreateClassDto[]) {
  // Verifica se o array veio vazio
  if (!createClassDto || createClassDto.length === 0) {
    throw new BadRequestException('Nenhuma classe foi enviada');
  }

  // Extrai todos os títulos enviados

  const titles = createClassDto.map((c) => c.title);
  
  // Busca no banco se já existe alguma classe com esses títulos
  const existingVideos = await this.classesRepository.find({
    where: { title: In(titles) }, // precisa importar In do TypeORM
  });

  if (existingVideos.length > 0) {
    const existingTitles = existingVideos.map((v) => v.title).join(', ');
    throw new ConflictException(
      `Já existe(m) vídeo(s) com o(s) título(s): ${existingTitles}`,
    );
  }

  const newClasses = this.classesRepository.create(createClassDto);
  return await this.classesRepository.save(newClasses);
}


  findAll() {
    return this.classesRepository.find();
  }

  findAllForManagement() {
    return this.classesRepository.find();
  }

  async findOneForManagement() {
    return this.classesRepository.find();
  }

  async findOne(videoId: number) {
    if (isNaN(videoId)) {
      throw new BadRequestException('ID inválido');
    }
    const video = await this.classesRepository.findOneBy({id: videoId});

    if (!video) { throw new NotFoundException("Ops. Vídeo não encontrado"); }

    return video;
  }


  async uploadImageToS3(file: Express.Multer.File, fileName: string): Promise<string> {
    const bucketName = process.env.AWS_S3_BUCKET!;

      //  const buffer = await sharp(file?.buffer)
              
      //         .toBuffer();
    

    const uploadParams = new PutObjectCommand({
      Bucket: bucketName,
      Key: fileName,
      Body: file.buffer,
      ContentType: file.mimetype,
    });

    console.log(uploadParams);

    try {
      await this.s3.send(uploadParams);
      return `https://${bucketName}.s3.${process.env.AWS_S3_REGION}.amazonaws.com/${fileName}`;
    } catch (err) {
      throw new BadRequestException('Erro ao enviar arquivo para o S3: ' + err);
    }
  }

  async update(updateClassDto: UpdateClassDto) {
    // 1. Converte a string JSON para objeto
    const dadosConvertidos = JSON.parse(updateClassDto['data']);
    
    // 2. Pega o objeto real com os dados (assumindo que é um array e o item é o primeiro)
    const dadosParaAtualizar = dadosConvertidos[0];
    const id = dadosParaAtualizar.id;

    // 3. Busca o vídeo existente
    const video = await this.findOne(id);

    if (!video) {
        throw new BadRequestException("Vídeo não encontrado.");
    }

    // CORREÇÃO AQUI: Merge com os dados convertidos, não com o DTO bruto
    const videoUpdated = this.classesRepository.merge(video, dadosParaAtualizar);

    try {
        await this.classesRepository.save(videoUpdated);
    } catch (error) {
        // Dica: Passe o erro original no segundo parâmetro para log ou trate a mensagem
        throw new BadRequestException(`Erro ao atualizar: ${error.message}`);
    }

    return { success: true, message: "Vídeo atualizado com sucesso", statusCode: 200 };
}

  async remove(videoId: number) {
    if (isNaN(videoId)) {
      throw new BadRequestException('ID inválido');
    }
    const video = await this.classesRepository.findOne({ where: { id: videoId } });
    if (!video) { throw new NotFoundException("Ops. Vídeo não encontrado"); }

    try { 
      await this.classesRepository.remove(video);
    } catch (error) {
      console.log(error);
      throw new BadRequestException("Erro ao tentar remover o vídeo. Erro", error);
    }

    return { message: "Vídeo removido com sucesso" }; 
  }

  /* ### Funções Auxiliares ### */

  async findClassById(classId: number) {
    const video = await this.classesRepository.findOneBy({id: classId});

    if (!video) {
      throw new NotFoundException("Vídeo não encontrado");
    }

    return video;
  }

  
}