import { BadRequestException, ConflictException, Injectable, NotFoundException } from '@nestjs/common';
import { CreateSignatureDto } from './dto/create-signature.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Signature } from './entities/signature.entity';
import { Repository } from 'typeorm';
import { UpdateSignatureDto } from './dto/update-signature.dto';
import { diskStorage } from 'multer';
import * as multerS3 from 'multer-s3';
import { S3Client, PutObjectCommand } from '@aws-sdk/client-s3';
import * as sharp from 'sharp';


@Injectable()
export class SignaturesService {
  constructor(
    @InjectRepository(Signature)
    private signatureRepository: Repository<Signature>

    
  ) {}

   private s3 = new S3Client({
    region: process.env.AWS_S3_REGION!,
    credentials: {
      accessKeyId: process.env.AWS_ACCESS_KEY_ID!,
      secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY!,
    },
  });



  async create(createSignatureDto: CreateSignatureDto) {

    const signature = await this.signatureRepository.findOne({
      where: { 
        signature: createSignatureDto.signature
      }
    });

    if (signature) {
      throw new ConflictException("Ops. Já existe uma assinatura com esse código");
    }

    

    return await this.signatureRepository.save(
      this.signatureRepository.create(createSignatureDto)
    );
  }

  findAll() {
    return this.signatureRepository.find();
  }

  findAllForManagement() {
    return this.signatureRepository.find();
  }

  async findOneForManagement() {
    return this.signatureRepository.find();
  }

  async findOne(signatureId: number) {
    const signature = await this.signatureRepository.findOneBy({id: signatureId});

    if (!signature) { throw new NotFoundException("Ops. Assinatura não encontrada"); }

    return signature;
  }


  async uploadImageToS3(file: Express.Multer.File, fileName: string): Promise<string> {
    const bucketName = process.env.AWS_S3_BUCKET!;

      //  const buffer = await sharp(file?.buffer)
              
      //         .toBuffer();
    

    const uploadParams = new PutObjectCommand({
      Bucket: bucketName,
      Key: fileName,
      Body: file.buffer,
      ContentType: file.mimetype,
    });

    console.log(uploadParams);

    try {
      await this.s3.send(uploadParams);
      return `https://${bucketName}.s3.${process.env.AWS_S3_REGION}.amazonaws.com/${fileName}`;
    } catch (err) {
      throw new BadRequestException('Erro ao enviar arquivo para o S3: ' + err);
    }
  }

  async update(signatureId: number, updateSignatureDto: UpdateSignatureDto) {
    const signature = await this.findOne(signatureId);

    const signatureUpdated = this.signatureRepository.merge(signature, updateSignatureDto);

    try { 
      await this.signatureRepository.save(signatureUpdated);
    } catch (error) {
      throw new BadRequestException("Erro ao tentar atualizar um ou mais dados da assinatura. Error: ", error);
    }

    return { success: true, message: "Assinatura atualizada com sucesso", statusCode: 200 };
  }

  async remove(signatureId: number) {
    const signature = await this.findOne(signatureId);

    try { 
      await this.signatureRepository.softDelete(signature.id);
    } catch (error) {
      throw new BadRequestException("Erro ao tentar remover a assinatura. Erro", error);
    }

    return { message: "Assinatura removida com sucesso" }; 
  }

  /* ### Funções Auxiliares ### */

  async findSignatureById(signatureId: number) {
    const signature = await this.signatureRepository.findOneBy({id: signatureId});

    if (!signature) {
      throw new NotFoundException("Assinatura não encontrada");
    }

    return signature;
  }
}