import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards, UseInterceptors, UploadedFile, UploadedFiles } from '@nestjs/common';
import { ProductsService } from './products.service';
import { CreateProductDto } from './dto/create-product.dto';
import { UpdateProductDto } from './dto/update-product.dto';
import { RequiredRoles } from 'src/shared/auth/required-roles.decorator';
import { AuthGuard } from 'src/shared/auth/auth.guard';
import { RoleGuard } from 'src/shared/auth/role/role.guard';
import { Roles } from 'src/shared/enums/enum';
import { FileInterceptor, FilesInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';


@Controller('produtos')
export class ProductsController {
  constructor(private readonly productsService: ProductsService) {}

@Post('/registro')
@UseInterceptors(FilesInterceptor('image')) // 'Files' no plural aceita o array que está chegando
async create(
  @Body() createProductDto: CreateProductDto, 
  @UploadedFiles() images: Express.Multer.File[] // Note o 's' e os colchetes []
) {
  let s3Url: string | null = null;

  // VERIFICAÇÃO: Checou se a lista 'images' existe e se tem pelo menos 1 arquivo
  if (images && images.length > 0) {
    
    // SOLUÇÃO DO PROBLEMA:
    // Pegamos apenas a primeira imagem da lista (índice 0) para salvar
    const mainImage = images[0]; 

    // Agora passamos 'mainImage', que é um arquivo único com buffer, para o S3
    s3Url = await this.productsService.uploadImageToS3(mainImage, mainImage.originalname);
  }

  // Atribui a URL (ou null se não tiver imagem) ao DTO
  createProductDto.image = s3Url ?? undefined;

  return this.productsService.create(createProductDto);
}

  @Get('/')
  findAll() {
    return this.productsService.findAll();
  }

  @Get('/restrito')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  findAllForManagement() {
    return this.productsService.findAllForManagement();
  }

  @Get(':id')
  findOne(@Param('id') id: string) {
    return this.productsService.findOne(+id);
  }

  @Get(':id/restrito')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  findOneForManagement() {
    return this.productsService.findOneForManagement();
  }

  @Patch(':id')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  @UseInterceptors(FileInterceptor('image'))
  async update(@Param('id') id: string, @Body() updateProductDto: UpdateProductDto, @UploadedFile() file: Express.Multer.File) {
    (file) ? updateProductDto.image = await this.productsService.uploadImageToS3(file, file.originalname) : null;
    return this.productsService.update(+id, updateProductDto);
  }

  @Delete(':id')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  remove(@Param('id') id: string) {
    return this.productsService.remove(+id);
  }
}
