import { Type } from 'class-transformer';
import {
  IsInt,
  IsNotEmpty,
  IsNumber,
  IsOptional,
  IsString,
  Max,
  MaxLength,
  Min,
} from 'class-validator';

export class CreateProductDto {
  @IsString({
    message: 'O código do produto é obrigatório e deve ser uma string válida',
  })
  cod_product: string;

  @IsString({
    message: 'O nome do produto é obrigatório e deve ser uma string válida',
  })
  name: string;

  @IsString({
    message:
      'A descrição do produto é obrigatória e deve ser uma string válida',
  })
  description: string;

  @Type(() => Number)
  @IsNotEmpty({ message: 'O preço é obrigatório' })
  @Min(1, { message: 'O preço do produto deve ser maior que 0.' })
  @Max(100000, { message: 'O preço do produto deve ser menor que R$100.000.' })
  price: number;

  @IsOptional()
  @IsString({
    message: 'O campo de imagem deve ser uma string (URL ou caminho local).',
  })
  @MaxLength(255, {
    message: 'O caminho/URL da imagem deve ter no máximo 255 caracteres.',
  })
  image?: string;

  @IsString({ message: 'A linha do produto deve ser uma string válida' })
  line: string;

  @IsNotEmpty({ message: 'A quantidade é obrigatória' })
  @Type(() => Number)
  @IsInt({ message: 'A quantidade deve ser um número inteiro.' })
  quantity: number;

  @IsString({ message: 'A categoria do produto deve ser uma string válida' })
  category: string;

  @IsInt({ message: 'O estoque de segurança deve ser um número inteiro.' })
  @Min(10, { message: 'O estoque de segurança deve ser no mínimo 10.' })
  @Type(() => Number)
  @IsNotEmpty({ message: 'O estoque de segurança é obrigatório' })
  safety_stock: number;

  @Type(() => Number)
  @IsInt({
    message: 'O tempo para substituição deve ser um número inteiro em horas.',
  })
  @Min(1, { message: 'O tempo para substituição deve ser no mínimo 1 hora.' })
  @Max(43800, {
    message:
      'O tempo para substituição não pode exceder 43.800 horas (5 anos).',
  })
  time_to_replacement: number;
}
