import { BadRequestException, ConflictException, forwardRef, Inject, Injectable, Logger, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { User } from 'src/shared/users/entities/user.entity';
import { Repository, DataSource } from 'typeorm';
import { Order } from './entities/order.entity';
import { OrderProduct } from 'src/ecommerce/orders/entities/order_product.entity';
import { OrderTransactions } from './entities/order_transaction.entity';
import { Status } from 'src/shared/enums/enum';
import { AsaasService } from 'src/ecommerce/asaas/asaas.service';
import { CheckoutDto, PaymentMethod } from './dto/checkout.dto';
import { UsersService } from 'src/shared/users/users.service';
import { Product } from '../products/entities/product.entity';
import { EventEmitter2, OnEvent } from '@nestjs/event-emitter';
import { emit } from 'process';
import { stat } from 'fs';
import { MailerService } from '@nestjs-modules/mailer';
import { Creditcard } from '../creditcard/entities/creditcard.entity';

@Injectable()
export class OrdersService {
  constructor(
    @InjectRepository(Order)
    private orderRepository: Repository<Order>,
  
    @InjectRepository(User)
    private userRepository: Repository<User>,

    @InjectRepository(OrderProduct)
    private orderProductRepository: Repository<OrderProduct>,

    @InjectRepository(OrderTransactions)
    private orderTransactionsRepository: Repository<OrderTransactions>,

    @InjectRepository(Product)
    private productRepository: Repository<Product>,

    @InjectRepository(Creditcard)
    private creditcardRepository: Repository<Creditcard>,

    @Inject(forwardRef(() => AsaasService))
    private readonly asaasService: AsaasService,
    private readonly usersService: UsersService,
    private readonly eventEmitter: EventEmitter2,
    private readonly mailerService: MailerService,
  ) {}
  async findAllByUser(currentUserId: number) {
    console.log(currentUserId);
    const orders = await this.orderRepository.find({
      where: {
        user: { id: currentUserId},
      }      
    })
 
 
    
    
    return (!orders) ? { message: "Nenhum pedido encontrado"} : orders;
  }

  async findAllProductsByOrderId(currentUserId: number, orderId: number) {
    const order = await this.orderRepository.findOne({
      where: {
        id: orderId,
        user: { id: currentUserId}
      },
      relations: ['order_product', 'order_product.product']
    });

    if (!order) {
      throw new NotFoundException(`Não foi possível localizar o pedido ${orderId} para o usuário ${currentUserId}`);
    }

    return order;
  }

  async findAll() {
    const orders = await this.orderRepository.find();

    return orders;
  }

  async findAllByStatus(status: Status) {
    const orders = await this.orderRepository.find({
      where: {
        status: status
      }
    })

    return orders
  }

  async restrictFindAllProductsByOrderId(orderId: number) {
    const orders = await this.orderRepository
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.order_product', 'orderProduct')
      .leftJoinAndSelect('orderProduct.product', 'product')
      .where('order.id = :orderId', {orderId})
      .getMany()

    return orders;
  }

  async checkout(currentUserId: number, checkoutDto: CheckoutDto) {
    const user = await this.usersService.findUserById(currentUserId);

    const cartExist = await this.findShoppingCartByUser(currentUserId);

    if (!cartExist) {
      throw new NotFoundException("Ops. Nenhum carrinho encontrado para esse usuário");
    }
    
    await this.orderRepository.update(cartExist.id, { 
      payment_method: checkoutDto.paymentMethod as any,
      status: Status.PENDING_PAYMENT,
    })
   const order = await   this.orderTransactionsRepository.create({
        order: cartExist,
      });
    await this.orderTransactionsRepository.save(order);

    const updatedOrder = await this.orderRepository.findOne({
      where: {
        id: cartExist.id
        },
      relations: ['user']
    });

    if (!updatedOrder) {
    throw new NotFoundException('Order not found after update.');
    }
    if (checkoutDto.paymentMethod === PaymentMethod.CREDIT_CARD) {

      const existCard= await this.creditcardRepository.findOne({
        where: { user: { id: currentUserId },
        credit_card_number: checkoutDto.credit_card_number},
      });

      if(!existCard){
        throw new Error("Cartão de crédito não encontrado");
      }






      const creditCard = await this.asaasService.createCreditCardPayment({user: user, value: updatedOrder.total_amount , orderId: updatedOrder.id, creditCard: existCard});
      
      await this.orderRepository.update(updatedOrder.id, {
        paymentId: creditCard.paymentId
      });
      await this.enviarEmailPedidoCriado(updatedOrder);
      this.eventEmitter.emit('order.placed', updatedOrder);
      this.eventEmitter.emit('notify.store', {cellphone: updatedOrder?.user.cellphone, message: "Seu pedido foi criado com sucesso e está aguardando pagamento!"});


    }else{

    const pix = await this.asaasService.createPixCharge({user: user, value: updatedOrder.total_amount , orderId: updatedOrder.id});
    
    await this.orderRepository.update(updatedOrder.id, {
      paymentId: pix.paymentId
    });
    await this.enviarEmailPedidoCriado(updatedOrder);
    this.eventEmitter.emit('order.placed', updatedOrder);
    this.eventEmitter.emit('notify.store', {cellphone: updatedOrder?.user.cellphone, message: "Seu pedido foi criado com sucesso e está aguardando pagamento!"});


    return pix;
    }


    //this.eventEmitter.emit('order.placed', updatedOrder);
    // this.eventEmitter.emit('notify.store', {cellphone: updatedOrder?.store.cellphone, message: "Há um novo pedido aguardando para ser aprovado"});

    //return this.asaasService.createPixCharge({user: user, value: cartExist.total_amount, orderId: cartExist.id});
  }

  async createQRcode(currentUserId: number){
    const order = await this.orderRepository.findOne({
      where: {
        user: { id: currentUserId},
        status: Status.PENDING_PAYMENT
      },
      relations: ['order_product', 'order_product.product']
    })

    if (!order) {
      throw new NotFoundException("Nenhum pedido pendente de pagamento encontrado para este usuário.");
    }
    const qrcode = await this.asaasService.getPixQrCode(order.paymentId)

 return {qrcode: qrcode, paymentId: order.paymentId};
  }

  async statusPaied(paymentId: string){
    const orders = await this.orderRepository.find({
      where: {
        paymentId: paymentId,
        status: Status.FINISHED
      }
    })
    if(!orders){
      return 'não pago'
    }

    return orders;
  }









  async changeOrderStatusById(id: number, status: Status) {
    const order = await this.orderRepository.findOneBy({id});

    if (!order) {
      throw new NotFoundException("Pedido não encontrado");
    }

    const orderStatusUpdated = order.status = status;
    await this.orderRepository.update(order.id, { status: orderStatusUpdated});

    return { message: "Status do pedido alterado com sucesso"}
  }

  /* ### Funções Auxiliares ### */

  async enviarEmailPedidoCriado(order: any) {
    await this.mailerService.sendMail({
      to: 'andrewfabiani24@gmail.com',//user.email,
      subject: 'Um novo pedido foi criado!',
      template: './email',
      context:{
        orderId: order.id,
        totalAmount: order.total_amount,
      }
    });
    return 'Email enviado com sucesso';
  }

  async findShoppingCartByUser(userId: number) {
    const rows = await this.orderRepository
      .createQueryBuilder('order')
      .innerJoin('order.order_product', 'orderProduct')
      .innerJoin('orderProduct.product', 'product')
      .innerJoin('order.user', 'user')
      .select([
        'order.id AS order_id',
        'order.status AS order_status',
        'order.total_amount AS order_total_amount',
        'order.total_items AS order_total_items',
        'user.id AS user_id',
        'orderProduct.id AS order_product_id',
        'orderProduct.quantity AS order_product_quantity',
        'orderProduct.unit_price AS order_product_unit_price',
      ])
      .where('order.status = :status AND user.id = :userId', {
        status: Status.OPEN,
        userId,
      })
      .getRawMany();

      if (!rows.length) return null;

    const order = {
      id: rows[0].order_id,
      status: rows[0].order_status,
      total_amount: rows[0].order_total_amount,
      total_items: rows[0].order_total_items,
      user_id: rows[0].user_id,
      products: rows.map((row) => ({
        id: row.order_product_id,
        quantity: row.order_product_quantity,
        unit_price: row.order_product_unit_price,
      })),
    };

    return order;
  }

  async findOrderById(orderId: number) {
    const itemsExist = await this.orderProductRepository.find({
      where: {
        order: { id: orderId}
      },
      relations: ['product']
    });

    if (!itemsExist) throw new NotFoundException("Nenhum produto encontrado para o pedido informado");

    return itemsExist;
  }
  // async validateQuantity(cartExist: any) {
  //   let messages: string[] = []
  //   for (let i = 0; i < cartExist.length; i++) {
  //     const product = await this.storeProductRepository.findOne({
  //       where: {
  //         id: cleanCart[i].id
  //       },
  //       relations: ['product'],
  //     })

  //     if (!product) {
  //       throw new NotFoundException("Ops. Produto não encontrado durante a validação de quantidade. Contate um administrador do sistema ou a loja responsável");
  //     }

  //     if (cleanCart[i].quantity > product.quantity_in_stock) {
  //       messages.push(
  //         `O produto ${product.product.name} não está disponível nessa quantidade. Restam apenas ${product.quantity_in_stock - product.safety_stock}`
  //       );
  //     }
  //   } 

  //   if (messages.length > 0) {
  //     throw new ConflictException(messages);
  //   }

  //   return null;
  // }

  @OnEvent('payment.confirmed')
  async verifyStock(orderId: {orderId: number}) {
    const items = await this.findOrderById(orderId.orderId);

    const productsName: String[] = []
    for (let item of items) {
      item.product.quantity = item.product.quantity - item.quantity
      await this.productRepository.save(item.product);

      if (item.product.quantity <= item.product.safety_stock) {
        productsName.push(item.product.name);
      }
    }

    if (productsName.length > 0) {
      this.eventEmitter.emit('lower.stock', {
        productsName
      })
    }
  }
}
