import { Injectable } from '@nestjs/common';
import { CreateDashboardDto } from './dto/create-dashboard.dto';
import { UpdateDashboardDto } from './dto/update-dashboard.dto';
import { User } from 'src/shared/users/entities/user.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Order } from '../orders/entities/order.entity';
import * as dayjs from 'dayjs';
import { Course_payment } from '../course_payment/entities/course_payment.entity';


@Injectable()
export class DashboardService {
  constructor(
    @InjectRepository(Order)
    private readonly orderRepository: Repository<Order>,
    @InjectRepository(Course_payment)
    private readonly coursePaymentRepository: Repository<Course_payment>,
    @InjectRepository(User)
    private readonly userRepository: Repository<User>,
  ) {}


  async findAllRecentTransactions() {
    // 1. Gerar os últimos 12 meses com dayjs
    const last12Months = Array.from({ length: 12 }).map((_, i) => {
      const date = dayjs().subtract(i, 'month');
      return {
        date: date.format('YYYY-MM'), // Para agrupar
        month: date.format('MMM/YY'), // Para exibir no gráfico
        total: 0,
      };
    }).reverse();

    const startDate = dayjs().subtract(11, 'month').startOf('month').format('YYYY-MM-DD');

    
    const [recentOrders, recentCoursePayments, totals, coursePaymentTotals, dailySales, dailyCoursePaymentSales] = await Promise.all([
      
      this.orderRepository.createQueryBuilder('order')
        .innerJoinAndSelect('order.user', 'user')
        .select([
          'order.id AS id',
          'user.user_name AS name',
          'user.email AS email',
          'order.status AS status',
          'order.created_at AS created_at',
          'order.payment_method AS payment_method',
          'order.total_amount AS total_amount',
        ])
        .addSelect("'order' AS transaction_type")
        .orderBy('order.created_at', 'DESC')
        .limit(10)
        .getRawMany(),

      // Consulta para transações de cursos recentes
      this.coursePaymentRepository.createQueryBuilder('course_payment')
        .innerJoinAndSelect('course_payment.user', 'user')
        .select([
            'course_payment.id as id', 'user.user_name as name', 'user.email as email', 'course_payment.status as status', 'course_payment.created_at as created_at', 'course_payment.amount as total_amount'
        ])
        .addSelect("'course_payment' AS transaction_type")
        .orderBy('course_payment.created_at', 'DESC')
        .limit(10)
        .getRawMany(),
      
      // Consulta para totais gerais (mês, dia, total)
      this.orderRepository.createQueryBuilder('order')
        .select([
          'SUM(order.total_amount) AS totalEcommerce',
          '(SELECT SUM(total_amount) FROM `order` WHERE DATE(created_at) = CURDATE()) AS totalDay',
          '(SELECT SUM(total_amount) FROM `order` WHERE YEAR(created_at) = YEAR(NOW()) AND MONTH(created_at) = MONTH(NOW()) AND status= "finalizado") AS totalMonthOrder'
        ])
        .getRawOne(),

      // Consulta para totais de pagamentos de curso
      this.coursePaymentRepository.createQueryBuilder('course_payment')
        .select([
          'SUM(course_payment.amount) AS totalCoursePayment',
          '(SELECT SUM(amount) FROM `course_payment` WHERE YEAR(created_at) = YEAR(NOW()) AND MONTH(created_at) = MONTH(NOW()) AND status = "PAID") AS totalMonthCourse'
        ])
        .getRawOne(),

      // 3. Consulta para vendas mensais nos últimos 12 meses
      this.orderRepository.createQueryBuilder('order')
        .select("DATE_FORMAT(order.created_at, '%Y-%m')", 'date')
        .addSelect('SUM(order.total_amount)', 'total')
        .where('order.created_at >= :startDate', { startDate })
        .andWhere('order.status = "finalizado"')
        .groupBy("DATE_FORMAT(order.created_at, '%Y-%m')")
        .orderBy('date', 'ASC')
        .getRawMany(),

      // Consulta para vendas mensais de cursos nos últimos 12 meses
      this.coursePaymentRepository.createQueryBuilder('course_payment')
        .select("DATE_FORMAT(course_payment.created_at, '%Y-%m')", 'date')
        .addSelect('SUM(course_payment.amount)', 'total')
        .where('course_payment.created_at >= :startDate', { startDate })
        .andWhere('course_payment.status = "PAID"')
        .groupBy("DATE_FORMAT(course_payment.created_at, '%Y-%m')")
        .orderBy('date', 'ASC')
        .getRawMany(),
    ]);

    // Merge and sort recent transactions
    const recentTransactions = [...recentOrders, ...recentCoursePayments]
      .sort((a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime())
      .slice(0, 10);
    
    // 4. Mapear os resultados das consultas de vendas mensais para o array de meses
    const combinedDailySales = [...dailySales, ...dailyCoursePaymentSales];
    const dailyTotals = combinedDailySales.reduce((acc, sale) => {
        const date = sale.date; // Já está no formato YYYY-MM
        acc[date] = (acc[date] || 0) + Number(sale.total);
        return acc;
    }, {});

    last12Months.forEach(month => {
        if (dailyTotals[month.date]) {
            month.total = dailyTotals[month.date];
        }
    })
    
    return {
      recentTransactions,
      
      totalMonth: Number(totals?.totalMonthOrder ?? 0) + Number(coursePaymentTotals?.totalMonthCourse ?? 0),
      totalEcommerce: Number(totals?.totalEcommerce ?? 0),
      totalEcommerceMonth: Number(totals?.totalMonthOrder ?? 0),
      totalCoursePaymentMonth: Number(coursePaymentTotals?.totalMonthCourse ?? 0),
      totalCoursePayment: Number(coursePaymentTotals?.totalCoursePayment ?? 0),
      last12MonthsSales: last12Months,
    };
  }

  async findAllTransactions(){
    
    const [recentOrders, recentCoursePayments, totals, coursePaymentTotals,] = await Promise.all([
      
      this.orderRepository.createQueryBuilder('order')
        .innerJoinAndSelect('order.user', 'user')
        .select([
          'order.id AS id',
          'user.user_name AS name',
          'user.email AS email',
          'order.status AS status',
          'order.created_at AS created_at',
          'order.payment_method AS payment_method',
          'order.total_amount AS total_amount',
        ])
        .addSelect("'order' AS transaction_type")
        .orderBy('order.created_at', 'DESC')
        .where('status = "finalizado"')
        //.limit(10)
        .getRawMany(),

      // Consulta para transações de cursos recentes
      this.coursePaymentRepository.createQueryBuilder('course_payment')
        .innerJoinAndSelect('course_payment.user', 'user')
        .select([
            'course_payment.id as id', 'user.user_name as name', 'user.email as email', 'course_payment.status as status', 'course_payment.created_at as created_at', 'course_payment.amount as total_amount'
        ])
        .addSelect("'course_payment' AS transaction_type")
        .orderBy('course_payment.created_at', 'DESC')
        .where('status = "PAID"')
        //.limit(10)
        .getRawMany(),
      
      // Consulta para totais gerais (mês, dia, total)
      this.orderRepository.createQueryBuilder('order')
        .select([
          'SUM(order.total_amount) AS totalEcommerce',
          '(SELECT SUM(total_amount) FROM `order` WHERE DATE(created_at) = CURDATE()) AS totalDay',
          '(SELECT SUM(total_amount) FROM `order` WHERE YEAR(created_at) = YEAR(NOW()) AND MONTH(created_at) = MONTH(NOW()) AND status= "finalizado") AS totalMonthOrder'
        ])
        .getRawOne(),

      // Consulta para totais de pagamentos de curso
      this.coursePaymentRepository.createQueryBuilder('course_payment')
        .select([
          'SUM(course_payment.amount) AS totalCoursePayment',
          '(SELECT SUM(amount) FROM `course_payment` WHERE YEAR(created_at) = YEAR(NOW()) AND MONTH(created_at) = MONTH(NOW()) AND status = "PAID") AS totalMonthCourse'
        ])
        .getRawOne(),


    ]);

    // Merge and sort recent transactions
    const recentTransactions = [...recentOrders, ...recentCoursePayments]
      .sort((a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime())
      //.slice(0, 10);
    

    
    return {
      recentTransactions,
      
      totalMonth: Number(totals?.totalMonthOrder ?? 0) + Number(coursePaymentTotals?.totalMonthCourse ?? 0),
      totalEcommerce: Number(totals?.totalEcommerce ?? 0),
      totalEcommerceMonth: Number(totals?.totalMonthOrder ?? 0),
      totalCoursePaymentMonth: Number(coursePaymentTotals?.totalMonthCourse ?? 0),
      totalCoursePayment: Number(coursePaymentTotals?.totalCoursePayment ?? 0),
    };
  }

  findOne(id: number) {
    return `This action returns a #${id} dashboard`;
  }

  update(id: number, updateDashboardDto: UpdateDashboardDto) {
    return `This action updates a #${id} dashboard`;
  }

  remove(id: number) {
    return `This action removes a #${id} dashboard`;
  }
}
