"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CourseService = void 0;
const common_2 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const client_s3_1 = require("@aws-sdk/client-s3");
const courses_entity_1 = require("./entites/courses.entity");
const class_entity_1 = require("../videos/entities/class.entity");
const user_entity_1 = require("../../shared/users/entities/user.entity");
const course_payment_entity_1 = require("../course_payment/entities/course_payment.entity");
const course_lesson_progress_entity_1 = require("./entites/course_lesson_progress.entity");
let CourseService = class CourseService {
    courseRepository;
    classRepository;
    userRepository;
    course_paymentRepository;
    courseLessonProgressRepository;
    constructor(courseRepository, classRepository, userRepository, course_paymentRepository, courseLessonProgressRepository) {
        this.courseRepository = courseRepository;
        this.classRepository = classRepository;
        this.userRepository = userRepository;
        this.course_paymentRepository = course_paymentRepository;
        this.courseLessonProgressRepository = courseLessonProgressRepository;
    }
    s3 = new client_s3_1.S3Client({
        region: process.env.AWS_S3_REGION,
        credentials: {
            accessKeyId: process.env.AWS_ACCESS_KEY_ID,
            secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
        },
    });
    async create(createCourseDto) {
        const { classIds, ...courseData } = createCourseDto;
        const classes = await this.classRepository.findBy({
            id: (0, typeorm_2.In)(classIds),
        });
        if (classes.length !== classIds.length) {
            const foundIds = classes.map((c) => c.id);
            const notFoundIds = classIds.filter((id) => !foundIds.includes(id));
            throw new common_2.NotFoundException(`Video aulas não encontradas: ${notFoundIds.join(', ')}`);
        }
        const allUrls = classes.map((classe) => classe.url);
        const course = this.courseRepository.create({
            ...courseData,
            urls: allUrls,
            classes: classes,
        });
        return await this.courseRepository.save(course);
    }
    async findAll() {
        const courses = await this.courseRepository
            .createQueryBuilder('course')
            .loadRelationCountAndMap('course.enrollments', 'course.payments', 'payment', (qb) => qb.where('payment.status = :status', { status: 'PAID' }))
            .leftJoinAndSelect('course.classes', 'classes')
            .getMany();
        return courses;
    }
    async findAllByUser(CurrentUser) {
        const query = this.courseRepository
            .createQueryBuilder('course')
            .select([
            'course.id AS id',
            'course.title AS title',
            'course.description AS description',
            'course.category AS category',
            'course.thumbnail AS thumbnail',
            'course.level AS level',
            'course.price AS price',
        ])
            .where('course.status = :status', { status: 'Publicado' })
            .addSelect('CASE WHEN payment.id IS NOT NULL THEN TRUE ELSE FALSE END', 'hasAccess')
            .leftJoin('course_payment', 'payment', 'payment.courseId = course.id AND payment.userId = :userId AND payment.status = :paymentStatus')
            .setParameters({
            status: 'Publicado',
            userId: CurrentUser,
            paymentStatus: 'PAID',
        });
        return await query.getRawMany();
    }
    async findOneByUser(courseId, CurrentUser) {
        const course = await this.courseRepository.findOne({
            select: {
                id: true,
                title: true,
                description: true,
                category: true,
                status: true,
                thumbnail: true,
                level: true,
                price: true,
            },
            where: { id: courseId },
            relations: ['classes'],
        });
        if (!course) {
            throw new common_2.NotFoundException('Curso não encontrado');
        }
        const payment = await this.course_paymentRepository.findOne({
            where: {
                course: { id: courseId },
                user: { id: CurrentUser },
                status: 'PAID',
            },
        });
        if (!payment) {
            return { ...course, hasAccess: false };
        }
        return { ...course, hasAccess: true };
    }
    findAllForManagement() {
        return this.courseRepository.find();
    }
    async findOneForManagement() {
        return this.courseRepository.find();
    }
    async findOne(courseId) {
        const course = await this.courseRepository.findOne({
            select: {
                id: true,
                title: true,
                description: true,
                category: true,
                status: true,
                thumbnail: true,
                level: true,
                price: true,
                classes: {
                    id: true,
                    title: true,
                    description: true,
                    url: true,
                },
            },
            where: { id: courseId },
            relations: ['classes'],
        });
        if (!course) {
            throw new common_2.NotFoundException('Curso não encontrado');
        }
        const { classes, ...rest } = course;
        const response = { ...rest, lessons: classes || [] };
        return response;
    }
    async uploadImageToS3(file, fileName) {
        const bucketName = process.env.AWS_S3_BUCKET;
        const uploadParams = new client_s3_1.PutObjectCommand({
            Bucket: bucketName,
            Key: fileName,
            Body: file.buffer,
            ContentType: file.mimetype,
        });
        console.log(uploadParams);
        try {
            await this.s3.send(uploadParams);
            return `https://${bucketName}.s3.${process.env.AWS_S3_REGION}.amazonaws.com/${fileName}`;
        }
        catch (err) {
            throw new common_2.BadRequestException('Erro ao enviar arquivo para o S3: ' + err);
        }
    }
    async update(courseId, updateCourseDto) {
        const { classIds, ...courseData } = updateCourseDto;
        const course = await this.courseRepository.findOne({
            where: { id: courseId },
            relations: ['classes'],
        });
        if (!course) {
            throw new common_2.NotFoundException(`Curso com ID ${courseId} não encontrado.`);
        }
        if (classIds && classIds.length > 0) {
            const classesEntities = await this.classRepository.findBy({
                id: (0, typeorm_2.In)(classIds),
            });
            if (classesEntities.length !== classIds.length) {
                throw new common_2.NotFoundException('Algumas aulas informadas não existem.');
            }
            course.classes = classesEntities;
            course.urls = classesEntities.map((c) => c.url);
        }
        this.courseRepository.merge(course, courseData);
        try {
            return await this.courseRepository.save(course);
        }
        catch (error) {
            console.error(error);
            throw new common_2.BadRequestException('Erro ao atualizar o curso.');
        }
    }
    async remove(courseId) {
        const course = await this.courseRepository.findOne({
            where: { id: courseId },
        });
        if (!course) {
            throw new common_2.NotFoundException(`Curso com ID ${courseId} não encontrado.`);
        }
        try {
            await this.courseRepository.remove(course);
        }
        catch (error) {
            console.log(error);
            throw new common_2.BadRequestException('Erro ao tentar remover o curso. Erro', error);
        }
        return { message: 'Curso removido com sucesso' };
    }
    async findCourseById(courseId) {
        const course = await this.courseRepository.findOneBy({ id: courseId });
        if (!course) {
            throw new common_2.NotFoundException('Curso não encontrado');
        }
        return course;
    }
    async findCourseWithClasses() {
        const course = await this.courseRepository.find({
            relations: ['classes'],
        });
        return course;
    }
    async findCourseByIdWithClasses(id) {
        const course = await this.courseRepository.find({
            where: { id },
            relations: ['classes'],
        });
        return course;
    }
    async completeLesson(courseId, classId, CurrentUser) {
        const iscompleted = await this.courseLessonProgressRepository.findOne({
            where: {
                user: { id: CurrentUser },
                course: { id: courseId },
                class: { id: classId },
                completed: true,
            },
        });
        if (iscompleted) {
            return { completed: true };
        }
        await this.courseLessonProgressRepository.save({
            course: { id: courseId },
            class: { id: classId },
            user: { id: CurrentUser },
            completed: true,
        });
        return { completed: true };
    }
    async courseProgress(CurrentUser, courseId) {
        const course = await this.courseRepository
            .createQueryBuilder('course')
            .select([
            'course.id AS id',
            'course.title AS title',
            'course.description AS description',
        ])
            .leftJoin('course_payment', 'payment', 'payment.courseId = course.id')
            .where('course.status = :status', { status: 'Publicado' })
            .andWhere('course.id = :courseId', { courseId })
            .andWhere('payment.userId = :userId', { userId: CurrentUser })
            .andWhere('payment.status = :paymentStatus', { paymentStatus: 'PAID' })
            .getRawOne();
        if (!course) {
            throw new Error('Curso não encontrado ou não pago');
        }
        const totalLessons = await this.classRepository.count({
            where: { courses: { id: courseId } },
        });
        const completedLessons = await this.courseLessonProgressRepository.count({
            where: {
                user: { id: CurrentUser },
                course: { id: courseId },
                completed: true,
            },
        });
        const progress = Math.round((completedLessons / totalLessons) * 100);
        return {
            ...course,
            totalLessons,
            completedLessons,
            progress,
            finished: completedLessons === totalLessons,
        };
    }
};
exports.CourseService = CourseService;
exports.CourseService = CourseService = __decorate([
    (0, common_2.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(courses_entity_1.Course)),
    __param(1, (0, typeorm_1.InjectRepository)(class_entity_1.Class)),
    __param(2, (0, typeorm_1.InjectRepository)(user_entity_1.User)),
    __param(3, (0, typeorm_1.InjectRepository)(course_payment_entity_1.Course_payment)),
    __param(4, (0, typeorm_1.InjectRepository)(course_lesson_progress_entity_1.CourseLessonProgress)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository])
], CourseService);
//# sourceMappingURL=courses.service.js.map